#!/usr/bin/env python2
#
# An XMLRPC-over-Jabber server implementation for Python.
# It's heavily borrowed from the excellent xmlrpclib.py
# and from Matthew Allum jrpc.py sample implementation.
#
#
# History: 
#          ver. 0.0.2+- 20060502  [GDR!] added optional custom name for registerMethod()
#          ver. 0.0.2 - 20041121  added Davin M.Potts patch to support
#                                 jabberpy v0.5.0 and client SSL connections
#          ver. 0.0.1 - 20031018  first public release.
#
#
# -----------------------
# Copyright 2003 by gian paolo ciceri <gp.ciceri@acm.org>
# under either GNU GPL/LGPL. All Rights Reserved.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of gian paolo ciceri
# not be used in advertising or publicity pertaining to distribution
# of the software without specific, written prior permission.
# GIAN PAOLO CICERI DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# GIAN PAOLO CICERI BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
# ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
# IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
# OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
#
#
"""

This class, using xmlrpclib, wraps an XML-RPC answer
on an iq with the namespace 'jabber:iq:rpc' and sends
it back to the calling jabber client, acting as a jrpc server.

For more info on this technique see http://www.pipetree.com/jabber/jrpc.html 

"""

import sys

## Import jabber module
#from jabber 
import jabber, xmlstream

## Import xmlrpclib - included in python2.2 
import xmlrpclib

# to read config file
import xml.dom.minidom


import thread, threading

class JabberXMLRPCServer:

    def __init__(self, con, maxjobs=5):
        self.con = con
        ## Set the callback to handle NS_RPC iq messages
        self.con.registerHandler('iq', self.iq_xmlrpc_CB, type='set', ns=jabber.NS_RPC)
        self.methods = {}
        # max number of concurrent methods
        self.semaphore = threading.Semaphore(maxjobs)
    
    def iq_xmlrpc_CB(self, con, iq):
        print "about to start thread", iq
        thread.start_new_thread(self.iq_xmlrpc_response_CB, (con, iq))
        
    
    ## This is called when an Iq is recieved 
    def iq_xmlrpc_response_CB(self, con, iq):

        ## Get the query part of the Iq and check its namespace and id
        IqID = iq.getID()
        print "IqID", IqID

        ## Get the querys 'payload' , will return an XMLStreanNode structure
        xmlrpc_node = iq.getQueryPayload()

        ## Let xmlrpclib parse the method call. The node becomes a string
        ## automatically when called in a str context. 
        params, method = xmlrpclib.loads("<?xml version='1.0'?>%s" % str(iq))
        print params, method
        
        res_iq = jabber.Iq(to=iq.getFrom(), type="set")

        ## Set the i'qs query namespace
        res_iq.setQuery(jabber.NS_RPC)

        ## Set the ID
        res_iq.setID(IqID)
        
        
        if method in self.methods.keys():
            try:
                ## limit the max.number of concurrent methods running ot maxjob
                self.semaphore.acquire()        
                result = self.methods[method](params)
                print "result ->", result
                xmlrpc_res = xmlrpclib.dumps( result , methodresponse=method)
            except Exception, e:
                xmlrpc_res = xmlrpclib.dumps( xmlrpclib.Fault(500, "method error:" + str(e)))
                self.semaphore.release()
            else:
                self.semaphore.release()
            
        else:
            xmlrpc_res = xmlrpclib.dumps( xmlrpclib.Fault(404, "method not found"))

        ## Set the query payload - can be an XML document or
        ## a Node structure
        
        res_iq.setQueryPayload(xmlrpc_res)
        con.send(res_iq)
        
        
        
    def registerMethod(self, method, name = None):
        if not name:
            name = method.__name__
        self.methods[name] = method
        print self.methods


def testMethod(params):
    import time
    print params[0]
    time.sleep(params[0])
    print "wake UP", params[0]
    return params


def sumMethod(params):
    print params
    return (params[0] + params[1],)


def readConfigFile(configFile):  
    dom = xml.dom.minidom.parse(configFile)
       
    _server = dom.getElementsByTagName("server")[0]
    _username = dom.getElementsByTagName("username")[0]
    _password = dom.getElementsByTagName("password")[0]
    _resource = dom.getElementsByTagName("resource")[0]    
    _port = dom.getElementsByTagName("port")[0]
    _connection = dom.getElementsByTagName("connection")[0]
    _maxjobs = dom.getElementsByTagName("maxjobs")[0]
        
    server   = str(_server.firstChild.nodeValue)
    port   = int(_port.firstChild.nodeValue)
    if str(_connection.firstChild.nodeValue) in ('TCP', 'TCP_SSL'):
        connection = eval('xmlstream.' + str(_connection.firstChild.nodeValue))
    else:
        connection = xmlstream.TCP
    username = str(_username.firstChild.nodeValue)
    password = str(_password.firstChild.nodeValue)
    resource = str(_resource.firstChild.nodeValue)
    maxjobs  = int(_maxjobs.firstChild.nodeValue)
    
    return (server, port, connection, username, password, resource, maxjobs)

def usage():
    print sys.argv[0], "usage: ", sys.argv[0], "configfile"
    sys.exit(-1)


if __name__ == '__main__':

    if (len(sys.argv) < 2):
       usage()

    ## Setup server and auth varibles
    server, port, connection, username, password, resource, maxjobs = readConfigFile(sys.argv[1])


    ## Get a jabber connection object, with logging to stderr
    con = jabber.Client(host=server, port=port, connection=connection, debug=0, log=0)


    ## Try and connect
    try:
       con.connect()
    except Exception, e:
       print "Couldn't connect: %s" % str(e)
       sys.exit(0)
    else:
       print "Connected"

    ## Authenticate
    if con.auth(username, password, resource):
        print "Authenticated"
    else:
        print "Auth failed", con.lastErr, con.lastErrCode
        sys.exit(1)
          
    ## Get the roster and send presence. Maybe not needed but
    ## it's a good practise.
    con.requestRoster()
    con.sendInitPresence()

    ## Initialize a RPC server
    jrpcServer = JabberXMLRPCServer(con, maxjobs)

    ## register methods
    jrpcServer.registerMethod(testMethod)
    jrpcServer.registerMethod(sumMethod)

    ## wait .... 
    while 1: con.process(1)

